#!/bin/bash
# 应用部署脚本（离线版本，开发模式）

APP_NAME="wuyuanbiaoba-web"
INSTALL_DIR="/opt/$APP_NAME"
SERVICE_NAME="$APP_NAME.service"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

echo "================================================"
echo "🚀 部署 $APP_NAME (离线开发模式)"
echo "================================================"

# Root 权限检查
if [ "$EUID" -ne 0 ]; then
    echo "❌ 错误: 需要 root 权限"
    echo "   使用: sudo ./deploy.sh"
    exit 1
fi

# systemd 检查
if ! pidof systemd &>/dev/null; then
    echo "❌ 错误: 系统不支持 systemd"
    exit 1
fi

# 检查 Node.js 环境
echo ""
echo "🔍 检查 Node.js 环境..."

if ! command -v node &> /dev/null; then
    echo "❌ 错误: 未安装 Node.js"
    echo "   请先安装 Node.js 20 或更高版本"
    exit 1
fi

if ! command -v npm &> /dev/null; then
    echo "❌ 错误: 未安装 npm"
    exit 1
fi

NODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)
if [ "$NODE_VERSION" -lt 20 ]; then
    echo "❌ 错误: Node.js 版本过低 ($(node -v))"
    echo "   需要 Node.js 20 或更高版本"
    exit 1
fi

echo "   ✅ Node.js: $(node -v)"
echo "   ✅ npm: $(npm -v)"

# 部署应用
echo ""
echo "🛠️  开始部署..."

# 停止服务
if systemctl is-active --quiet $SERVICE_NAME; then
    echo "   停止服务..."
    systemctl stop $SERVICE_NAME
    sleep 2
fi

if systemctl is-enabled --quiet $SERVICE_NAME 2>/dev/null; then
    echo "   禁用服务..."
    systemctl disable $SERVICE_NAME 2>/dev/null
fi

# 清理旧备份（保留7天内的）
if [ -d "/opt" ]; then
    echo "   清理旧备份..."
    find /opt -maxdepth 1 -name "${APP_NAME}_backup_*" -type d -mtime +7 -exec rm -rf {} \; 2>/dev/null || true
fi

# 备份现有版本
if [ -d "$INSTALL_DIR" ]; then
    BACKUP_DIR="${INSTALL_DIR}_backup_$(date +%Y%m%d_%H%M%S)"
    echo "   备份到: $BACKUP_DIR"
    mv "$INSTALL_DIR" "$BACKUP_DIR"
fi

# 创建安装目录
mkdir -p $INSTALL_DIR

# 复制应用文件
echo "   复制文件..."

# 创建文件清单，确保只复制打包时包含的文件
REQUIRED_FILES=(
    "package.json"
    "package-lock.json"
    "index.html"
    "config.cjs"
)

REQUIRED_DIRS=(
    "client"
    "server"
    "script"
    "node_modules"
)

# 复制必需文件
for file in "${REQUIRED_FILES[@]}"; do
    if [ -f "$SCRIPT_DIR/$file" ]; then
        cp "$SCRIPT_DIR/$file" "$INSTALL_DIR/"
        echo "      ✅ $file"
    else
        echo "      ⚠️  警告: 缺少 $file"
    fi
done

# 复制必需目录
for dir in "${REQUIRED_DIRS[@]}"; do
    if [ -d "$SCRIPT_DIR/$dir" ]; then
        cp -r "$SCRIPT_DIR/$dir" "$INSTALL_DIR/"
        echo "      ✅ $dir/"
    else
        echo "      ⚠️  警告: 缺少 $dir"
    fi
done

# 复制可选文件
[ -f "$SCRIPT_DIR/README.md" ] && cp "$SCRIPT_DIR/README.md" "$INSTALL_DIR/" && echo "      ✅ README.md"

# 验证关键文件
echo ""
echo "🔍 验证部署..."

MISSING=0
REQUIRED_CHECK=(
    "package.json"
    "node_modules"
    "server"
    "client"
    "index.html"
    "config.cjs"
)

for item in "${REQUIRED_CHECK[@]}"; do
    if [ ! -e "$INSTALL_DIR/$item" ]; then
        echo "   ❌ 缺少: $item"
        MISSING=1
    else
        echo "   ✅ $item"
    fi
done

if [ $MISSING -eq 1 ]; then
    echo ""
    echo "❌ 部署验证失败"
    exit 1
fi

# 验证关键依赖
if [ ! -d "$INSTALL_DIR/node_modules/@peace/react_client" ]; then
    echo "   ⚠️  警告: 缺少关键依赖 @peace/react_client"
fi

# 获取 npm 和 node 的完整路径
NPM_PATH=$(which npm)
NODE_PATH=$(which node)

echo ""
echo "⚙️  配置服务..."
echo "   Node: $NODE_PATH"
echo "   npm: $NPM_PATH"

# 设置 systemd 环境变量
SYSTEMD_PATH="/usr/local/bin:/usr/local/sbin:/usr/sbin:/usr/bin:/sbin:/bin"

# 创建启动脚本
cat > $INSTALL_DIR/start.sh << STARTEOF
#!/bin/bash
cd $INSTALL_DIR
export NODE_ENV=development
export PATH=$SYSTEMD_PATH
exec $NPM_PATH start
STARTEOF

chmod +x $INSTALL_DIR/start.sh

# 创建 systemd 服务文件（开发模式）
cat > /etc/systemd/system/$SERVICE_NAME << SERVICE_EOF
[Unit]
Description=$APP_NAME Node.js Application
After=network.target

[Service]
Type=simple
User=root
WorkingDirectory=$INSTALL_DIR
ExecStart=$INSTALL_DIR/start.sh
Restart=on-failure
RestartSec=10
Environment=NODE_ENV=development
Environment=PATH=$SYSTEMD_PATH

StandardOutput=journal
StandardError=journal

[Install]
WantedBy=multi-user.target
SERVICE_EOF

# 重新加载 systemd
systemctl daemon-reload

# 启用并启动服务
systemctl enable $SERVICE_NAME
systemctl start $SERVICE_NAME

# 等待服务启动
sleep 3

echo ""
echo "================================================"
echo "📊 部署结果"
echo "================================================"
systemctl status $SERVICE_NAME --no-pager

# 清理部署目录（如果在 /tmp 下）
DEPLOY_DIR="$SCRIPT_DIR"
if [[ "$DEPLOY_DIR" == "/tmp/"* ]]; then
    echo ""
    echo "🗑️  清理部署临时目录: $DEPLOY_DIR"
    cd /tmp
    rm -rf "$DEPLOY_DIR"
fi

echo ""
echo "================================================"
echo "✅ 部署完成!"
echo "================================================"
echo ""
echo "📌 常用命令:"
echo "   状态: systemctl status $SERVICE_NAME"
echo "   日志: journalctl -u $SERVICE_NAME -f"
echo "   重启: systemctl restart $SERVICE_NAME"
echo "   停止: systemctl stop $SERVICE_NAME"
echo ""
