#!/bin/bash
# 远程打包脚本 - 在 Windows Git Bash 中运行
# 功能: 上传代码到 Linux 服务器，在 Linux 上打包，然后下载离线包

# ===== 配置区域 - 请修改为你的 Linux 服务器信息 =====
REMOTE_USER="root"
REMOTE_HOST="10.8.30.179"  # 修改为你的服务器地址或IP
REMOTE_PORT="22"
REMOTE_WORK_DIR="/tmp/wuyuanbiaoba-pack"
# ========================================================

APP_NAME="wuyuanbiaoba-web"

# 读取版本号
if [ -f package.json ]; then
    VERSION=$(grep '"version"' package.json | head -1 | sed -E 's/.*"version": *"([^"]+)".*/\1/')
    [ -z "$VERSION" ] && VERSION="latest"
else
    VERSION="latest"
fi

DEPLOY_PACKAGE="${APP_NAME}-${VERSION}-offline.tar.gz"

echo "================================================"
echo "🌐 远程 Linux 打包"
echo "================================================"
echo "目标服务器: $REMOTE_USER@$REMOTE_HOST"
echo "工作目录:   $REMOTE_WORK_DIR"
echo "应用版本:   $VERSION"
echo ""

# 检查配置
if [ "$REMOTE_HOST" = "your-linux-server.com" ]; then
    echo "❌ 错误: 请先配置服务器信息"
    echo ""
    echo "编辑文件: script/pack-remote.sh"
    echo "修改以下配置:"
    echo "  REMOTE_USER=\"root\""
    echo "  REMOTE_HOST=\"192.168.1.100\"  # 你的 Linux 服务器"
    echo "  REMOTE_PORT=\"22\""
    echo ""
    exit 1
fi

# 检查 SSH 连接
echo "🔍 检查服务器连接..."
if ! ssh -p $REMOTE_PORT -o ConnectTimeout=5 $REMOTE_USER@$REMOTE_HOST "echo '连接成功'" 2>/dev/null; then
    echo "❌ 无法连接到服务器"
    echo ""
    echo "请检查:"
    echo "  1. 服务器地址是否正确: $REMOTE_HOST"
    echo "  2. SSH 端口是否正确: $REMOTE_PORT"
    echo "  3. SSH 密钥是否配置"
    echo "  4. 网络是否通畅"
    echo ""
    echo "测试连接命令:"
    echo "  ssh -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST"
    echo ""
    echo "配置 SSH 免密登录:"
    echo "  ssh-keygen -t rsa -b 4096"
    echo "  ssh-copy-id -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST"
    echo ""
    exit 1
fi

echo "✅ 服务器连接正常"
echo ""

# 创建远程工作目录
echo "📁 准备远程工作目录..."
ssh -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST "mkdir -p $REMOTE_WORK_DIR"

# 上传源代码（排除不必要的文件）
echo ""
echo "📤 打包源代码..."

# 创建临时 tar 包（排除不必要的文件）
TEMP_TAR="/tmp/${APP_NAME}-source-$$.tar.gz"
tar -czf "$TEMP_TAR" \
    --exclude='node_modules' \
    --exclude='build' \
    --exclude='.git' \
    --exclude='*.tar.gz' \
    --exclude='.npm' \
    --exclude='.cache' \
    --exclude='.vscode' \
    --exclude='*.log' \
    ./

if [ $? -ne 0 ]; then
    echo "❌ 打包失败"
    exit 1
fi

SOURCE_SIZE=$(du -h "$TEMP_TAR" | cut -f1)
echo "✅ 源代码打包完成 ($SOURCE_SIZE)"

echo ""
echo "📤 上传到服务器..."
echo "   (这可能需要一些时间...)"

scp -P $REMOTE_PORT "$TEMP_TAR" $REMOTE_USER@$REMOTE_HOST:$REMOTE_WORK_DIR/source.tar.gz

if [ $? -ne 0 ]; then
    echo ""
    echo "❌ 上传失败"
    rm -f "$TEMP_TAR"
    exit 1
fi

echo "✅ 上传完成"

# 删除本地临时文件
rm -f "$TEMP_TAR"

# 在服务器上解压
echo ""
echo "📦 在服务器上解压源代码..."
ssh -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST "cd $REMOTE_WORK_DIR && tar -xzf source.tar.gz && rm -f source.tar.gz"

if [ $? -ne 0 ]; then
    echo "❌ 解压失败"
    exit 1
fi

echo "✅ 源代码准备完成"

# 在远程服务器执行打包
echo ""
echo "================================================"
echo "🔨 在 Linux 服务器上执行打包..."
echo "================================================"
echo ""

ssh -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST "cd $REMOTE_WORK_DIR && bash script/package-app-offline.sh"

if [ $? -ne 0 ]; then
    echo ""
    echo "❌ 远程打包失败"
    echo ""
    echo "你可以手动登录服务器检查:"
    echo "  ssh -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST"
    echo "  cd $REMOTE_WORK_DIR"
    echo "  bash script/package-app-offline.sh"
    exit 1
fi

# 下载打包文件
echo ""
echo "📥 下载打包文件到本地..."

scp -P $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST:$REMOTE_WORK_DIR/$DEPLOY_PACKAGE ./

if [ $? -ne 0 ]; then
    echo "❌ 下载失败"
    exit 1
fi

echo ""
echo "✅ 打包文件已下载到本地"

# 清理远程临时文件（可选）
echo ""
read -p "🗑️  是否清理远程临时文件? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo "   清理中..."
    ssh -p $REMOTE_PORT $REMOTE_USER@$REMOTE_HOST "rm -rf $REMOTE_WORK_DIR"
    echo "   ✅ 已清理"
else
    echo "   保留远程文件: $REMOTE_USER@$REMOTE_HOST:$REMOTE_WORK_DIR"
fi

LOCAL_SIZE=$(du -h $DEPLOY_PACKAGE | cut -f1)

echo ""
echo "================================================"
echo "🎉 远程打包完成!"
echo "================================================"
echo ""
echo "📦 本地文件: $DEPLOY_PACKAGE ($LOCAL_SIZE)"
echo ""
echo "📋 下一步:"
echo "   1. 将此文件上传到目标离线服务器"
echo "   2. 在服务器上解压并部署:"
echo "      tar -xzf $DEPLOY_PACKAGE"
echo "      sudo ./deploy.sh"
echo ""
