import { useState, useEffect } from 'react';

// localStorage 存储键名
const TEMPLATE_STORAGE_KEY = 'wuyuanbiaoba_templates';

// 将图片转换为base64的工具函数
const convertImageToBase64 = (imagePath) => {
   return new Promise((resolve, reject) => {
      const img = new Image();
      img.crossOrigin = 'anonymous'; // 处理跨域问题

      img.onload = () => {
         const canvas = document.createElement('canvas');
         const ctx = canvas.getContext('2d');

         canvas.width = img.width;
         canvas.height = img.height;

         ctx.drawImage(img, 0, 0);

         try {
            const base64 = canvas.toDataURL('image/jpeg', 0.8);
            resolve(base64);
         } catch (error) {
            reject(error);
         }
      };

      img.onerror = () => {
         reject(new Error(`Failed to load image: ${imagePath}`));
      };

      img.src = imagePath;
   });
};

// 创建内置模板的函数
const createBuiltinTemplate = async () => {
   try {
      // 标靶图片路径
      const imagePath = '/client/assets/img/wuyuanbiaoba_template_images/temp_target1.jpg';
      const imageBase64 = await convertImageToBase64(imagePath);

      return {
         key: 'builtin_1',
         name: '标靶模板1',
         isBaseline: false,
         isPerspectiveCorrection: false,
         binaryThreshold: 100,
         gaussianBlur: 1,
         physicalRadius: 40.0,
         gradientThresholdValue: 100,
         anchorThresholdValue: 80,
         imageUrl: imageBase64,
         id: 'builtin_1',
      };
   } catch (error) {
      console.error('创建内置模板失败:', error);
      // 如果图片加载失败，返回不带图片的内置模板
      return {
         key: 'builtin_1',
         name: '标靶模板1',
         isBaseline: false,
         isPerspectiveCorrection: false,
         binaryThreshold: 100,
         gaussianBlur: 1,
         physicalRadius: 40.0,
         gradientThresholdValue: 100,
         anchorThresholdValue: 80,
         imageUrl: "",
         id: 'builtin_1',
      };
   }
};

// localStorage 工具函数
const saveTemplatesToStorage = (templates) => {
   try {
      localStorage.setItem(TEMPLATE_STORAGE_KEY, JSON.stringify(templates));
      return true;
   } catch (error) {
      console.error('保存模板数据到localStorage失败:', error);
      return false;
   }
};

const getTemplatesFromStorage = () => {
   try {
      const stored = localStorage.getItem(TEMPLATE_STORAGE_KEY);
      return stored ? JSON.parse(stored) : null;
   } catch (error) {
      console.error('从localStorage读取模板数据失败:', error);
      return null;
   }
};

// 模板存储管理 Hook
export const useTemplateStorage = () => {
   const [templates, setTemplates] = useState([]);
   const [loading, setLoading] = useState(true);

   // 初始化加载模板数据
   useEffect(() => {
      const loadTemplates = async () => {
         const storedTemplates = getTemplatesFromStorage();

         if (storedTemplates && storedTemplates.length > 0) {
            // 如果localStorage中有数据，检查是否已有内置模板
            const hasBuiltinTemplate = storedTemplates.some(template => template.key === 'builtin_1');

            if (!hasBuiltinTemplate) {
               // 如果没有内置模板，添加内置模板
               try {
                  const builtinTemplate = await createBuiltinTemplate();
                  const updatedTemplates = [builtinTemplate, ...storedTemplates];
                  setTemplates(updatedTemplates);
                  saveTemplatesToStorage(updatedTemplates);
                  console.log('内置模板已添加到现有模板列表');
               } catch (error) {
                  console.error('添加内置模板失败:', error);
                  setTemplates(storedTemplates);
               }
            } else {
               // 已有内置模板，直接使用现有数据
               setTemplates(storedTemplates);
            }
         } else {
            // 如果localStorage中没有数据，只创建内置模板
            try {
               const builtinTemplate = await createBuiltinTemplate();
               const allTemplates = [builtinTemplate];

               setTemplates(allTemplates);
               saveTemplatesToStorage(allTemplates);
               console.log('已创建内置模板');
            } catch (error) {
               console.error('创建内置模板失败:', error);
               // 如果创建失败，创建空模板列表
               setTemplates([]);
               saveTemplatesToStorage([]);
            }
         }

         setLoading(false);
      };

      loadTemplates();
   }, []);

   // 添加模板
   const addTemplate = (templateInfo) => {
      const newTemplates = [...templates, templateInfo];
      setTemplates(newTemplates);
      const saved = saveTemplatesToStorage(newTemplates);

      if (!saved) {
         // 如果保存失败，回滚状态
         console.error('添加模板失败：无法保存到localStorage');
         return false;
      }

      return true;
   };

   // 更新模板
   const updateTemplate = (templateInfo) => {
      const updatedTemplates = templates.map((item) =>
         item.key === templateInfo.key ? templateInfo : item
      );
      setTemplates(updatedTemplates);
      const saved = saveTemplatesToStorage(updatedTemplates);

      if (!saved) {
         // 如果保存失败，回滚状态
         console.error('更新模板失败：无法保存到localStorage');
         setTemplates(templates); // 回滚到原状态
         return false;
      }

      return true;
   };

   // 删除模板
   const deleteTemplate = (templateKey) => {
      const filteredTemplates = templates.filter((item) => item.key !== templateKey);
      setTemplates(filteredTemplates);
      const saved = saveTemplatesToStorage(filteredTemplates);

      if (!saved) {
         // 如果保存失败，回滚状态
         console.error('删除模板失败：无法保存到localStorage');
         setTemplates(templates); // 回滚到原状态
         return false;
      }

      return true;
   };

   // 根据key查找模板
   const getTemplateByKey = (key) => {
      return templates.find(template => template.key === key);
   };

   // 清空所有模板
   const clearAllTemplates = () => {
      setTemplates([]);
      const saved = saveTemplatesToStorage([]);

      if (!saved) {
         console.error('清空模板失败：无法保存到localStorage');
         return false;
      }

      return true;
   };

   // 重置为默认模板
   const resetToDefaultTemplates = async () => {
      try {
         const builtinTemplate = await createBuiltinTemplate();
         const allTemplates = [builtinTemplate];

         setTemplates(allTemplates);
         const saved = saveTemplatesToStorage(allTemplates);

         if (!saved) {
            console.error('重置模板失败：无法保存到localStorage');
            return false;
         }

         return true;
      } catch (error) {
         console.error('重置模板失败:', error);
         return false;
      }
   };

   return {
      templates,
      loading,
      addTemplate,
      updateTemplate,
      deleteTemplate,
      getTemplateByKey,
      clearAllTemplates,
      resetToDefaultTemplates,
   };
};

export default useTemplateStorage;
