import { useState, useEffect, useCallback } from "react";

const CORRECT_PASSWORD_HASH = "77796ac7e66ecc44954287ed7de7096c4016dd6ffb2763091c4eb3bc4d28b6dc";
const CORRECT_PASSWORD_HASH_FOR_LINUX = "dd77d88a991f5d0b9ff689d4a26306c08f49be64866eaa0af409d459a40b1780";
const AUTH_STATUS_KEY = "advanced_config_unlocked";

/**
 * 纯 JS SHA-256 实现
 */
function sha256Pure(str) {
  const K = [
    0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
    0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
    0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
    0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
    0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
    0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
    0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
    0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
  ];

  function rightRotate(value, amount) {
    return (value >>> amount) | (value << (32 - amount));
  }

  const msg = new TextEncoder().encode(str);
  const msgLen = msg.length * 8;

  // 填充消息
  const paddedLen = Math.ceil((msgLen + 65) / 512) * 512 / 8;
  const padded = new Uint8Array(paddedLen);
  padded.set(msg);
  padded[msg.length] = 0x80;

  // 添加长度（大端序）
  for (let i = 0; i < 8; i++) {
    padded[paddedLen - 1 - i] = (msgLen >>> (i * 8)) & 0xff;
  }

  // 初始哈希值
  let h0 = 0x6a09e667, h1 = 0xbb67ae85, h2 = 0x3c6ef372, h3 = 0xa54ff53a;
  let h4 = 0x510e527f, h5 = 0x9b05688c, h6 = 0x1f83d9ab, h7 = 0x5be0cd19;

  // 处理每个 512 位块
  for (let chunk = 0; chunk < paddedLen; chunk += 64) {
    const w = new Uint32Array(64);

    // 前 16 个字（大端序）
    for (let i = 0; i < 16; i++) {
      w[i] = (padded[chunk + i * 4] << 24) |
             (padded[chunk + i * 4 + 1] << 16) |
             (padded[chunk + i * 4 + 2] << 8) |
             padded[chunk + i * 4 + 3];
    }

    // 扩展到 64 个字
    for (let i = 16; i < 64; i++) {
      const s0 = rightRotate(w[i - 15], 7) ^ rightRotate(w[i - 15], 18) ^ (w[i - 15] >>> 3);
      const s1 = rightRotate(w[i - 2], 17) ^ rightRotate(w[i - 2], 19) ^ (w[i - 2] >>> 10);
      w[i] = (w[i - 16] + s0 + w[i - 7] + s1) >>> 0;
    }

    let a = h0, b = h1, c = h2, d = h3, e = h4, f = h5, g = h6, h = h7;

    for (let i = 0; i < 64; i++) {
      const S1 = rightRotate(e, 6) ^ rightRotate(e, 11) ^ rightRotate(e, 25);
      const ch = (e & f) ^ (~e & g);
      const temp1 = (h + S1 + ch + K[i] + w[i]) >>> 0;
      const S0 = rightRotate(a, 2) ^ rightRotate(a, 13) ^ rightRotate(a, 22);
      const maj = (a & b) ^ (a & c) ^ (b & c);
      const temp2 = (S0 + maj) >>> 0;

      h = g;
      g = f;
      f = e;
      e = (d + temp1) >>> 0;
      d = c;
      c = b;
      b = a;
      a = (temp1 + temp2) >>> 0;
    }

    h0 = (h0 + a) >>> 0;
    h1 = (h1 + b) >>> 0;
    h2 = (h2 + c) >>> 0;
    h3 = (h3 + d) >>> 0;
    h4 = (h4 + e) >>> 0;
    h5 = (h5 + f) >>> 0;
    h6 = (h6 + g) >>> 0;
    h7 = (h7 + h) >>> 0;
  }

  // 转为十六进制字符串
  return [h0, h1, h2, h3, h4, h5, h6, h7]
    .map(h => h.toString(16).padStart(8, '0'))
    .join('');
}

async function generatePasswordHash(password) {
  try {
    const normalized = String(password || "").trim();

    // 尝试使用 Web Crypto API
    if (typeof crypto !== 'undefined' && crypto.subtle && crypto.subtle.digest) {
      const data = new TextEncoder().encode(normalized);
      const hashBuffer = await crypto.subtle.digest("SHA-256", data);
      return Array.from(new Uint8Array(hashBuffer))
        .map(b => b.toString(16).padStart(2, "0"))
        .join("");
    }

    // 回退到纯 JS 实现
    console.warn("crypto.subtle 不可用，使用纯 JS SHA-256 实现");
    return sha256Pure(normalized);
  } catch (e) {
    console.error("密码哈希生成失败:", e);
    throw e;
  }
}

function checkUnlockStatus() {
  return sessionStorage.getItem(AUTH_STATUS_KEY) === "true";
}

function setUnlockStatus(unlocked) {
  if (unlocked) {
    sessionStorage.setItem(AUTH_STATUS_KEY, "true");
  } else {
    sessionStorage.removeItem(AUTH_STATUS_KEY);
  }
}

function useAuth() {
  const [isUnlocked, setIsUnlocked] = useState(checkUnlockStatus);

  useEffect(() => {
    setIsUnlocked(checkUnlockStatus());
  }, []);

  return {
    isUnlocked,
    verifyPassword: useCallback(async (password) => {
      try {
        if (!password) return false;

        const hash = await generatePasswordHash(password);
        const isValid = (hash === CORRECT_PASSWORD_HASH || hash === CORRECT_PASSWORD_HASH_FOR_LINUX);

        if (isValid) {
          setUnlockStatus(true);
          setIsUnlocked(true);
        }

        return isValid;
      } catch (e) {
        console.error("密码验证失败:", e);
        return false;
      }
    }, []),
    logout: useCallback(() => {
      setUnlockStatus(false);
      setIsUnlocked(false);
    }, [])
  };
}

export { useAuth, generatePasswordHash, checkUnlockStatus };
