const net = require('net');
const WebSocket = require('ws');
// TCP代理配置
let TCP_HOST = '127.0.0.1'; // 因为下位机和上位机在同一台机器上，所以使用localhost
const TCP_PORT = 2230;
// 创建独立的WebSocket服务器用于TCP代理
function setupTcpProxy(conf) {
   if (conf && conf.flag === 'localdev') {
      TCP_HOST = '10.8.30.179' //本地开发配置
   }
   console.log(`TCP代理目标地址: ${TCP_HOST}:${TCP_PORT}`);
   // 从配置中获取端口，如果没有则使用默认端口
   const wsPort = (conf && conf.port) ? Number(conf.port) + 1 : 5001; // WebSocket端口比HTTP端口大1

   // console.log(`准备在端口 ${wsPort} 启动WebSocket服务器`);

   const wss = new WebSocket.Server({
      port: wsPort,
      host: '0.0.0.0', // 监听所有网络接口，允许局域网访问
      path: '/tcp-proxy'
   });

   wss.on('connection', (ws, request) => {
      // console.log(`WebSocket连接建立，来自: ${request.socket.remoteAddress}`);

      // 创建TCP连接
      const tcpClient = new net.Socket();
      // TCP数据缓冲区
      let tcpDataBuffer = '';
      let tcpConnected = false; // 新增：TCP连接状态标志

      // TCP连接成功
      tcpClient.connect(process.env.TCP_PORT || TCP_PORT, process.env.TCP_HOST || TCP_HOST, () => {
         // console.log(process.env);
         // console.log(`TCP连接已建立到 ${TCP_HOST}:${TCP_PORT}`);
         tcpConnected = true;

         // 向WebSocket客户端发送就绪信号
         if (ws.readyState === WebSocket.OPEN) {
            const readySignal = JSON.stringify({
               _from: 'proxy',
               cmd: 'ready',
               values: {
                  message: 'TCP connection established',
                  timestamp: Date.now()
               }
            });
            ws.send(readySignal);
         }
      });

      // TCP接收数据，转发到WebSocket
      tcpClient.on('data', (data) => {
         let textData;
         // 尝试解析为文本
         try {
            textData = data.toString('utf8');
         } catch (e) {
            console.log('TCP数据无法解析为文本');
            return;
         }

         // 将新数据添加到缓冲区
         tcpDataBuffer += textData;

         // 检查是否有完整的消息（以\n\n结尾）
         let endIndex;
         while ((endIndex = tcpDataBuffer.indexOf('\n\n')) !== -1) {
            // 提取完整消息
            const completeMessage = tcpDataBuffer.substring(0, endIndex);
            // 从缓冲区移除已处理的消息
            tcpDataBuffer = tcpDataBuffer.substring(endIndex + 2);

            // console.log('提取到完整消息:', completeMessage.length, '字节');

            // 转发完整消息到WebSocket
            if (ws.readyState === WebSocket.OPEN) {
               // console.log('准备发送完整消息到WebSocket:', completeMessage.length, '字节');
               // console.log('消息内容:', completeMessage);
               ws.send(completeMessage, (err) => {
                  if (err) {
                     console.error(`[${new Date().toLocaleString()}] WebSocket发送数据错误:`, err);
                  } else {
                     // console.log('完整消息已转发到WebSocket客户端');
                  }
               });
            }
         }

         // console.log('缓冲区剩余数据:', tcpDataBuffer.length, '字节');
      });

      // WebSocket接收数据，转发到TCP
      ws.on('message', (data) => {
         // 处理可能的Buffer数据，转换为字符串
         let messageStr;

         if (Buffer.isBuffer(data)) {
            messageStr = data.toString('utf8');
            // console.log('WebSocket数据(Buffer转字符串):', messageStr);
         } else if (typeof data === 'string') {
            messageStr = data;
            // console.log('WebSocket数据(字符串):', messageStr);
         } else {
            // console.warn('收到未知类型数据，已忽略:', typeof data);
            return;
         }

         // 转发字符串数据到TCP服务器
         if (tcpClient.writable) {
            console.log('发送数据到TCP服务器:', messageStr);

            // 检查数据大小
            const dataSize = Buffer.byteLength(messageStr, 'utf8');
            // console.log(`数据大小: ${dataSize} bytes`);

            // 直接发送完整数据
            tcpClient.write(messageStr + '\n\n', (err) => {
               if (err) {
                  console.error(`[${new Date().toLocaleString()}] TCP发送数据错误:`, err);
               } else {
                  // console.log('数据已发送到TCP服务器');
               }
            });
         } else {
            // console.warn('TCP连接不可写，无法发送数据');
         }
      });

      // TCP连接错误处理
      tcpClient.on('error', (err) => {
         console.error(`[${new Date().toLocaleString()}] TCP连接错误:`, err);
         tcpDataBuffer = ''; // 清理缓冲区
         if (ws.readyState === WebSocket.OPEN) {
            ws.close(1011, 'TCP连接错误');
         }
      });

      // TCP连接关闭
      tcpClient.on('close', () => {
         // console.log('TCP连接已关闭');
         tcpDataBuffer = ''; // 清理缓冲区
         if (ws.readyState === WebSocket.OPEN) {
            ws.close();
         }
      });

      // WebSocket连接关闭
      ws.on('close', () => {
         // console.log('WebSocket连接已关闭');
         tcpDataBuffer = ''; // 清理缓冲区
         if (tcpClient.writable) {
            tcpClient.destroy();
         }
      });

      // WebSocket错误处理
      ws.on('error', (err) => {
         console.error(`[${new Date().toLocaleString()}] WebSocket错误:`, err);
         if (tcpClient.writable) {
            tcpClient.destroy();
         }
      });
   });

   wss.on('listening', () => {
      // console.log(`TCP代理WebSocket服务器已启动在端口 ${wsPort}，路径: /tcp-proxy`);
      // console.log(`局域网连接地址: ws://[本机IP]:${wsPort}/tcp-proxy`);
      // console.log(`本地连接地址: ws://localhost:${wsPort}/tcp-proxy`);
      // console.log(`注意：请确保防火墙允许端口 ${wsPort} 的访问`);
   });

   wss.on('error', (err) => {
      console.error(`[${new Date().toLocaleString()}] WebSocket服务器错误:`, err);
   });

   return wss;
}


module.exports = { setupTcpProxy };
